-- Implementering af et semantisk tableau
module Tableau where

import Data.Set (Set)
import qualified Data.Set as Set
import Data.Tree (Tree)
import qualified Data.Tree as Tree
import Language

freeEquiv = ((Atom "p") `Implies` (Atom "q")) `And` ((Atom "q") `Implies` (Atom "p"))

-- Repræsentation af en sekvent og et tableau
type Sequent = (FormulaSet, FormulaSet)
type System  = [Sequent]
type Tableau = Tree System

-- Find formeltyper
findFormulaType    :: (Formula -> Bool) -> FormulaSet -> Maybe Formula
findFormulaType p fs
                | hasFormula = Just (Set.findMin formulas)
                | otherwise  = Nothing               
                  where
                    formulas   = Set.filter p fs
                    hasFormula = Set.size formulas > 0

-- Specifikke funktioner til at finde bestemte formeltyper
findDisjunction, findConjunction, findImplication :: (Formula -> Bool) -> FormulaSet -> Maybe Formula
findConjunction p fs = findFormulaType (\x->(getFormulaType x) == "Conjunction" && p x) fs
findDisjunction p fs = findFormulaType (\x->(getFormulaType x) == "Disjunction" && p x) fs
findImplication p fs = findFormulaType (\x->(getFormulaType x) == "Implication" && p x) fs

-- Definer typen af "et tilfælde" / tableaukonstruktionsregel
type Constructor    = System -> Maybe [System]

-- Prædikat der angiver om et givet tilfælde (c) virker på et givet system (sys)
caseApplies        :: Constructor -> System -> Bool
caseApplies c sys   = case c sys of
                      Just _  -> True
                      Nothing -> False

-- Tilfælde / Tableaukonstruktionsregler
tAnd, fAnd, tOr, fOr, tImpl, fImpl :: Constructor
tAnd sys
      | meets_cond = Just [sys'] -- Et enkelt system
      | otherwise  = Nothing
        where
          meets_cond    = length validsequents > 0
          cond g        = (\(And p q)-> (p `Set.notMember` g) || (q `Set.notMember` g))  -- Sidebetingelse
          -- Find de sekventer, der opfylder betingelserne
          validsequents = [ ((g,d),u) | (g,d) <- sys, let u = findConjunction (cond g) g, u /= Nothing ]
          ((sequent),u) = head validsequents
          (gamma,delta) = sequent
          -- Udtræk de direkte deludsagn fra konjunktionen
          (Just (And p q)) = u
          -- Konstruér det nye system
          sequent'      = (Set.insert q $ Set.insert p $ gamma, delta)
          sys'          = sequent' : (filter (\x->not (x == sequent)) sys)
          
fAnd sys
      | meets_cond = Just (s1 : s2 : []) -- Liste af to systemer
      | otherwise  = Nothing
        where
          meets_cond       = length validsequents > 0
          cond d           = (\(And p q)-> (p `Set.notMember` d) && (q `Set.notMember` d))  -- Sidebetingelse
          -- Find de sekventer, der opfylder betingelserne
          validsequents    = [ ((g,d),u) | (g,d) <- sys, let u = findConjunction (cond d) d, u /= Nothing ]
          (sequent,u)      = head validsequents
          (gamma,delta)    = sequent
          -- Udtræk de direkte deludsagn fra konjunktionen
          (Just (And p q)) = u
          -- Konstruér de to nye systemer
          (sq1,sq2)        = ((gamma, Set.insert p delta), (gamma, Set.insert q delta))
          sys'             = filter (\x->not (x == sequent)) sys
          (s1,s2)          = (sq1 : sys', sq2 : sys')

tOr sys
      | meets_cond = Just (s1 : s2 : [])
      | otherwise  = Nothing
        where
          meets_cond      = length validsequents > 0
          cond g          = (\(Or p q)-> (p `Set.notMember` g) && (q `Set.notMember` g))  -- Sidebetingelse
          -- Find de sekventer, der opfylder betingelserne
          validsequents   = [ ((g,d),u) | (g,d) <- sys, let u = findDisjunction (cond g) g, u /= Nothing ]
          (sequent,u)     = head validsequents
          (gamma,delta)   = sequent
          -- Udtræk de direkte deludsagn fra disjunktionen
          (Just (Or p q)) = u
          -- Konstruér de to nye systemer
          (sq1,sq2)       = ((Set.insert p gamma, delta), (Set.insert q gamma, delta))
          sys'            = filter (\x->not (x == sequent)) sys
          (s1,s2)         = (sq1 : sys', sq2 : sys')
          
fOr sys
      | meets_cond = Just [sys']
      | otherwise  = Nothing
        where
          meets_cond       = length validsequents > 0
          cond d           = (\(Or p q)-> (p `Set.notMember` d) && (q `Set.notMember` d))  -- Sidebetingelse
          -- Find de sekventer, der opfylder betingelserne
          validsequents    = [ ((g,d),u) | (g,d) <- sys, let u = findDisjunction (cond d) d, u /= Nothing ]
          (sequent,u)      = head validsequents
          (gamma,delta)    = sequent
          -- Udtræk de direkte deludsagn fra disjunktionen
          (Just (Or p q))  = u
          -- Konstruér det nye system
          sequent'         = (gamma, Set.insert q $ Set.insert p delta)
          sys'             = sequent' : (filter (\x->not (x == sequent)) sys)
          
tImpl sys
      | meets_cond = Just (s1 : s2 : [])
      | otherwise  = Nothing
        where
          meets_cond          = length validsequents > 0
          cond g d            = (\(Implies a b)-> (a `Set.notMember` d) && (b `Set.notMember` g))  -- Sidebetingelse
          -- Find de sekventer, der opfylder betingelserne
          validsequents       = [ ((g,d),u) | (g,d) <- sys, let u = findImplication (cond g d) g, u /= Nothing ]
          (sequent,u)         = head validsequents
          (gamma,delta)       = sequent
          -- Udtræk de direkte deludsagn fra implikationen
          (Just (Implies p q))= u
          -- Konstruér de to nye systemer
          (sq1,sq2)           = ((gamma, Set.insert p delta), (Set.insert q gamma, delta))
          sys'                = filter (\x->not (x == sequent)) sys
          (s1,s2)             = (sq1 : sys', sq2 : sys')

fImpl sys
      | meets_cond = Just [sys']
      | otherwise  = Nothing
        where
          meets_cond  = length validsequents > 0
          cond g d (Implies a b)= length [ (g',d') | (g',d') <- sys, ((Set.insert a g) `Set.isSubsetOf` g') || (b `Set.member` d') ] == 0
          -- Find de sekventer, der opfylder betingelserne
          validsequents         = [ ((g,d),u) | (g,d) <- sys, let u = findImplication (cond g d) d, u /= Nothing ]
          -- Tag den første sekvent fra listen
          (sequent,u)           = head validsequents 
          (gamma,delta)         = sequent
          -- Udtræk de direkte deludsagn fra implikationen
          (Just (Implies p q))  = u
          -- Konstruér den nye sekvent og tilføj den til systemet
          sequent'              = (Set.insert p gamma, Set.singleton q)
          sys'                  = sequent' : sys

-- Tableaukonstruktion
tableau   :: Formula -> Tableau
tableau f  = construct [(Set.empty, Set.singleton f)]
          
construct :: System -> Tableau
construct sys
          | notLeaf   = Tree.Node sys (map construct systems)
          | otherwise = Tree.Node sys [] -- [] er det tomme undertræ.
            where 
              result  = applyCases sys
              notLeaf = result /= Nothing
              (Just systems) = result

-- Prøver hvert tilfælde én efter én
applyCases :: Constructor
applyCases sys 
         | caseApplies tAnd sys  = tAnd sys
         | caseApplies fAnd sys  = fAnd sys                   
         | caseApplies tOr  sys  = tOr sys
         | caseApplies fOr  sys  = fOr sys                         
         | caseApplies tImpl sys = tImpl sys
         | caseApplies fImpl sys = fImpl sys
         | otherwise             = Nothing

-- Prædikat der angiver om en sekvent er lukket
isClosed :: Sequent -> Bool
isClosed sq = (gamma `Set.intersection` delta) /= Set.empty
              where (gamma,delta) = sq

-- Prædikat der angiver om et tableau er vellykket
isSuccessful :: Tableau -> Bool
isSuccessful (Tree.Node sys subforest)
                     | subforest == [] = any isClosed sys
                     | otherwise       = all isSuccessful subforest